<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Event\EventInterface;

/**
 * Users Controller
 *
 * @property \App\Model\Table\UsersTable $Users
 */
class UsersController extends AppController
{
    public function beforeFilter(EventInterface $event): void
    {
        parent::beforeFilter($event);
        $this->Authentication->addUnauthenticatedActions(['login']);
    }

    public function beforeRender(EventInterface $event)
    {
        parent::beforeRender($event);
        $this->viewBuilder()->setLayout('login');
    }

    public function login()
    {
        $this->request->allowMethod(['get', 'post']);
        $result = $this->Authentication->getResult();

        // Check if the user is logged in successfully
        if ($result && $result->isValid()) {
            // Get authenticated user data
            $user = $result->getData();

            // Check if the user account is active
            if ($user['status_id'] == 1) {
                // Log successful login attempt
                $this->log('User ' . $user['email'] . ' logged in successfully.', 'info');

                // Redirect to home page or to the requested URL
                $redirect = $this->request->getQuery('redirect', [
                    'controller' => 'Home',
                    'action' => 'index',
                ]);

                return $this->redirect($redirect);
            } else {
                // Account is inactive or locked, display error
                $this->Flash->error(__('Account is not activated, Please activate your account'));
                return;
            }
        }

        // Handle failed login attempts
        if ($this->request->is('post')) {
            if (!$result->isValid()) {
                // Log failed login attempt
                $this->log('Failed login attempt for username: ' . $this->request->getData('username'), 'warning');

                // Show error message
                $this->Flash->error(__('Your username or password is incorrect.'));
            }
        }
    }

     public function logout()
    {
        $result = $this->Authentication->getResult();

        if ($result && $result->isValid()) {
            $this->Authentication->logout();
            $this->Flash->success('You are now logged out.');
            return $this->redirect(['controller' => 'Users', 'action' => 'login']);
        } else {
            $this->Flash->info('You are not logged in.');
            return $this->redirect(['controller' => 'Users', 'action' => 'login']);
        }
    }

}
