<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query\SelectQuery;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * PageContents Model
 *
 * @property \App\Model\Table\MenusTable&\Cake\ORM\Association\BelongsTo $Menus
 * @property \App\Model\Table\ContentsTable&\Cake\ORM\Association\BelongsTo $Contents
 *
 * @method \App\Model\Entity\PageContent newEmptyEntity()
 * @method \App\Model\Entity\PageContent newEntity(array $data, array $options = [])
 * @method array<\App\Model\Entity\PageContent> newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\PageContent get(mixed $primaryKey, array|string $finder = 'all', \Psr\SimpleCache\CacheInterface|string|null $cache = null, \Closure|string|null $cacheKey = null, mixed ...$args)
 * @method \App\Model\Entity\PageContent findOrCreate($search, ?callable $callback = null, array $options = [])
 * @method \App\Model\Entity\PageContent patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method array<\App\Model\Entity\PageContent> patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\PageContent|false save(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method \App\Model\Entity\PageContent saveOrFail(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method iterable<\App\Model\Entity\PageContent>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\PageContent>|false saveMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\PageContent>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\PageContent> saveManyOrFail(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\PageContent>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\PageContent>|false deleteMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\PageContent>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\PageContent> deleteManyOrFail(iterable $entities, array $options = [])
 */
class PageContentsTable extends Table
{
    /**
     * Initialize method
     *
     * @param array<string, mixed> $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('page_contents');
        $this->setDisplayField('id');
        $this->setPrimaryKey('id');

        $this->belongsTo('Menus', [
            'foreignKey' => 'menu_id',
        ]);
        $this->belongsTo('Contents', [
            'foreignKey' => 'content_id',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->integer('menu_id')
            ->allowEmptyString('menu_id');

        $validator
            ->integer('content_id')
            ->allowEmptyString('content_id');

        return $validator;
    }

    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->existsIn(['menu_id'], 'Menus'), ['errorField' => 'menu_id']);
        $rules->add($rules->existsIn(['content_id'], 'Contents'), ['errorField' => 'content_id']);

        return $rules;
    }

    public function getRelatedMenusWithContents(int $menuId)
    {
        return $this->find('all')
            ->contain([
                'Contents.ContentImages',
                'Menus.ParentMenus'
            ])
            ->where([
                'OR' => [
                    ['Menus.id' => $menuId],
                    ['Menus.parent_menu_id' => $menuId],
                    ['ParentMenus.parent_menu_id' => $menuId]
                ]
            ])
            ->order([
                'Menus.name' => 'ASC',
                'Contents.name' => 'ASC'
            ]);
    }

    public function findRelatedMenusBySlug($slug, $id)
    {
        return $this->find()
            ->contain([
                'Contents.ContentImages',
                'Menus.ParentMenus'
            ])
            ->where([
                'OR' => [
                    ['Menus.slug' => $slug],
                    ['ParentMenus.slug' => $slug],
                    ['ParentMenus.parent_menu_id' => $id]
                ]
            ])
            ->order([
                'Menus.name' => 'ASC',
                'Contents.id' => 'DESC'
            ]);
    }

    public function findPendingListings($slug, $id)
    {
        return $this->find()
            ->contain([
                'Contents.ContentImages',
                'Menus.ParentMenus'
            ])
            ->where([
                'Contents.status' => 0,
                'OR' => [
                    ['Menus.slug' => $slug],
                    ['ParentMenus.slug' => $slug],
                    ['ParentMenus.parent_menu_id' => $id]
                ]
            ])
            ->order([
                'Menus.name' => 'ASC',
                'Contents.id' => 'DESC'
            ]);

    }

    public function findEnableListings($slug, $id)
    {
        return $this->find()
            ->contain([
                'Contents.ContentImages',
                'Menus.ParentMenus'
            ])
            ->where([
                'Contents.status' => 1,
                'OR' => [
                    ['Menus.slug' => $slug],
                    ['ParentMenus.slug' => $slug],
                    ['ParentMenus.parent_menu_id' => $id]
                ]
            ])
            ->order([
                'Menus.name' => 'ASC',
                'Contents.id' => 'DESC'
            ]);

    }

    public function findDisableListings($slug, $id)
    {
        return $this->find()
            ->contain([
                'Contents.ContentImages',
                'Menus.ParentMenus'
            ])
            ->where([
                'Contents.status' => 99,
                'OR' => [
                    ['Menus.slug' => $slug],
                    ['ParentMenus.slug' => $slug],
                    ['ParentMenus.parent_menu_id' => $id]
                ]
            ])
            ->order([
                'Menus.name' => 'ASC',
                'Contents.id' => 'DESC'
            ]);

    }

    public function findEditedListings($slug, $id)
    {
        return $this->find()
            ->contain([
                'Contents.ContentImages',
                'Menus.ParentMenus'
            ])
            ->where([
                'Contents.status' => 2,
                'OR' => [
                    ['Menus.slug' => $slug],
                    ['ParentMenus.slug' => $slug],
                    ['ParentMenus.parent_menu_id' => $id]
                ]
            ])
            ->order([
                'Menus.name' => 'ASC',
                'Contents.id' => 'DESC'
            ]);
    }

    public function pageContents($id)
    {
        return $this->find()
            ->where(["content_id" => $id])
            ->first();
    }
    
}
