<?php
declare(strict_types=1);

/**
 * CakePHP(tm) : Rapid Development Framework (https://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 * @link      https://cakephp.org CakePHP(tm) Project
 * @since     3.3.0
 * @license   https://opensource.org/licenses/mit-license.php MIT License
 */
namespace App;

use Cake\Core\Configure;
use Cake\Core\ContainerInterface;
use Cake\Datasource\FactoryLocator;
use Cake\Error\Middleware\ErrorHandlerMiddleware;
use Cake\Http\BaseApplication;
use Cake\Http\Middleware\BodyParserMiddleware;
use Cake\Http\Middleware\CsrfProtectionMiddleware;
use Cake\Http\MiddlewareQueue;
use Cake\ORM\Locator\TableLocator;
use Cake\Routing\Middleware\AssetMiddleware;
use Cake\Routing\Middleware\RoutingMiddleware;
use Cake\Http\Middleware\SecurityHeadersMiddleware;
use Cake\Http\Middleware\HttpsEnforcerMiddleware;
use Cake\Http\Middleware\CspMiddleware;

use Authentication\AuthenticationService;
use Authentication\AuthenticationServiceInterface;
use Authentication\AuthenticationServiceProviderInterface;
use Authentication\Middleware\AuthenticationMiddleware;
use Cake\Routing\Router;
use Psr\Http\Message\ServerRequestInterface;
/**
 * Application setup class.
 *
 * This defines the bootstrapping logic and middleware layers you
 * want to use in your application.
 */
class Application extends BaseApplication implements AuthenticationServiceProviderInterface
{
    /**
     * Load all the application configuration and bootstrap logic.
     *
     * @return void
     */
    public function bootstrap(): void
    {
        // Call parent to load bootstrap from files.
        parent::bootstrap();

        if (PHP_SAPI === 'cli') {
            $this->bootstrapCli();
        } else {
            FactoryLocator::add(
                'Table',
                (new TableLocator())->allowFallbackClass(false)
            );
        }

        /*
         * Only try to load DebugKit in development mode
         * Debug Kit should not be installed on a production system
         */
        if (Configure::read('debug')) {
            $this->addPlugin('DebugKit');
        }

        // Load more plugins here
    }

    /**
     * Setup the middleware queue your application will use.
     *
     * @param \Cake\Http\MiddlewareQueue $middlewareQueue The middleware queue to setup.
     * @return \Cake\Http\MiddlewareQueue The updated middleware queue.
     */
    public function middleware(MiddlewareQueue $middlewareQueue): MiddlewareQueue
    {

        $policy = [
            // Default source directives
            'default-src' => [
                'self' => true,
            ],

            // Script sources
            'script-src' => [
                'allow' => [
                    'https://www.google-analytics.com',
                    'https://www.googletagmanager.com',
                    'https://cdn.jsdelivr.net',
                    'https://cdnjs.cloudflare.com',
                    'https://unpkg.com',
                    'https://cdn.tiny.cloud',
                    'https://cdn.ckeditor.com',
                ],
                'self' => true,
                'unsafe-inline' => false,
                'unsafe-eval' => false,
                'strict-dynamic' => true,
            ],

            // Style sources
            'style-src' => [
                'allow' => [
                    'https://fonts.googleapis.com',
                    'https://cdn.jsdelivr.net',
                    'https://cdnjs.cloudflare.com',
                    'https://unpkg.com',
                    'https://cdn.tiny.cloud',
                    'https://cdn.ckeditor.com',
                ],
                'self' => true,
                'unsafe-inline' => false, // Often needed for dynamic styles
            ],

            // Image sources
            'img-src' => [
                'allow' => [
                    'https:',
                    'https://unpkg.com',
                    'https://cdn.jsdelivr.net',
                    'https://cdnjs.cloudflare.com',
                    'data:', // For inline images              
                ],
                'self' => true,
            ],

            // Font sources
            'font-src' => [
                'allow' => [
                    'https://fonts.gstatic.com',
                    'https://cdnjs.cloudflare.com',
                    'https://cdn.jsdelivr.net',
                    'data:',
                ],
                'self' => true,
            ],

            // Connect sources (XHR, WebSocket, etc.)
            'connect-src' => [
                'allow' => [
                    'https://www.google-analytics.com',
                    'https://stats.g.doubleclick.net',
                    'https://api.example.com'
                ],
                'self' => true,
            ],

            // Form action destinations
            'form-action' => [
                'self' => true,
                'allow' => [
                    'https://your-payment-gateway.com',
                ],
            ],

            // Frame sources
            'frame-src' => [
                'allow' => [],
                'self' => true,
            ],

            // Object sources (plugins, embeds)
            'object-src' => [
                'self' => true,
                'allow' => [],
            ],

            // Media sources (audio, video)
            'media-src' => [
                'self' => true,
                'allow' => [],
            ],

            // Frame ancestors (embedding control)
            'frame-ancestors' => [
                'self' => true,
                'allow' => [],
            ],

            // Worker sources
            'worker-src' => [
                'self' => true,
                'allow' => [],
            ],

            // Manifest sources
            'manifest-src' => [
                'self' => true,
            ],

            // Base URI restriction
            'base-uri' => [
                'self' => true,
            ],
        ];
        // Enable automatic nonce addition to script & CSS link tags.
        $csp = new CspMiddleware($policy, [
            'scriptNonce' => true,
            'styleNonce' => true,
        ]);

        $securityHeaders = new SecurityHeadersMiddleware();
        $securityHeaders
            ->setCrossDomainPolicy()
            ->setReferrerPolicy()
            ->setXFrameOptions()
            ->setXssProtection()
            ->noOpen()
            ->noSniff();

        // Send a 302 status code when redirecting
        $https = new HttpsEnforcerMiddleware([
            'redirect' => true,
            'statusCode' => 302,
        ]);

        $https = new HttpsEnforcerMiddleware([
            'hsts' => [
                // How long the header value should be cached for.
                'maxAge' => 60 * 60 * 24 * 365,
                // should this policy apply to subdomains?
                'includeSubDomains' => true,
                // Should the header value be cacheable in google's HSTS preload
                // service? While not part of the spec it is widely implemented.
                'preload' => true,
            ],
        ]);

        $middlewareQueue
            // Catch any exceptions in the lower layers,
            // and make an error page/response
            ->add(new ErrorHandlerMiddleware(Configure::read('Error'), $this))

            // Handle plugin/theme assets like CakePHP normally does.
            ->add(new AssetMiddleware([
                'cacheTime' => Configure::read('Asset.cacheTime'),
            ]))

            // Add routing middleware.
            // If you have a large number of routes connected, turning on routes
            // caching in production could improve performance.
            // See https://github.com/CakeDC/cakephp-cached-routing
            ->add(new RoutingMiddleware($this))

            // Parse various types of encoded request bodies so that they are
            // available as array through $request->getData()
            // https://book.cakephp.org/4/en/controllers/middleware.html#body-parser-middleware
            ->add(new BodyParserMiddleware())

            // Cross Site Request Forgery (CSRF) Protection Middleware
            // https://book.cakephp.org/4/en/security/csrf.html#cross-site-request-forgery-csrf-middleware
            ->add(new CsrfProtectionMiddleware([
                // Cookie configuration
                'cookieName' => 'csrfToken',
                'expiry' => '+1 hour',
                'secure' => true,
                'httponly' => true,
                'samesite' => 'Strict',
            ]))

            ->add(new AuthenticationMiddleware($this))

            /**
             * Adds SecurityHeadersMiddleware to apply security-related HTTP headers.
             *
             * This middleware sets headers like `X-Frame-Options`, `X-XSS-Protection`, 
             * and `X-Content-Type-Options` to enhance application security.
             *
             * @param \SecurityHeadersMiddleware $securityHeaders Middleware for HTTP security headers.
             * @return void
             */
            ->add($securityHeaders)


            /**
             * Adds HttpsEnforcerMiddleware to redirect HTTP requests to HTTPS.
             *
             * This middleware ensures that all traffic is securely transmitted over HTTPS, 
             * improving the security of data in transit and protecting against man-in-the-middle attacks.
             *
             * @param \HttpsEnforcerMiddleware $https Middleware to enforce HTTPS connections.
             * @return void
             */
            ->add($https)


            /**
             * Adds CspMiddleware to enforce Content Security Policy (CSP) settings.
             *
             * This middleware restricts the sources for scripts and styles, helping to prevent 
             * cross-site scripting (XSS) attacks and ensuring that only trusted content is loaded.
             *
             * @param \CspMiddleware $csp Middleware to implement content security policies.
             * @return void
             */
            ->add($csp)
        ;


        return $middlewareQueue;
    }

    public function getAuthenticationService(ServerRequestInterface $request): AuthenticationServiceInterface
    {

        $authenticationService = new AuthenticationService([
            'unauthenticatedRedirect' => Router::url('/'),
            'queryParam' => 'redirect',
        ]);

        // Load identifiers, ensure we check email and password fields
        $authenticationService->loadIdentifier('Authentication.Password', [
            'fields' => [
                'username' => 'email',
                'password' => 'password',
            ],
            'userModel' => 'Users'
        ]);

        // Load the authenticators, you want session first
        $authenticationService->loadAuthenticator('Authentication.Session');
        // Configure form data check to pick email and password
        $authenticationService->loadAuthenticator('Authentication.Form', [
            'fields' => [
                'username' => 'email',
                'password' => 'password',
            ],
            'loginUrl' => Router::url('/'),
        ]);

        return $authenticationService;
    }

    /**
     * Register application container services.
     *
     * @param \Cake\Core\ContainerInterface $container The Container to update.
     * @return void
     * @link https://book.cakephp.org/4/en/development/dependency-injection.html#dependency-injection
     */
    public function services(ContainerInterface $container): void
    {
    }

    /**
     * Bootstrapping for CLI application.
     *
     * That is when running commands.
     *
     * @return void
     */
    protected function bootstrapCli(): void
    {
        $this->addOptionalPlugin('Bake');

        $this->addPlugin('Migrations');

        // Load more plugins here
    }
}
